class MessageQueue {
    constructor() {
        this.queue = [];
        this.processing = false;
        this.handler = null;
        // 添加超时时间配置，默认为4秒
        this.timeout = 60000; // 毫秒
        this.defaultRequestOptions = {
            method: 'POST',
            url: 'https://example.com',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        };
    }

    // 设置超时时间（可选，默认4秒）
    setTimeout(timeoutMs) {
        if (typeof timeoutMs === 'number' && timeoutMs > 0) {
            this.timeout = timeoutMs;
        } else {
            throw new Error('Timeout must be a positive number');
        }
    }

    setHandler(handler) {
        if (typeof handler !== 'function') {
            throw new Error('Handler must be a function');
        }
        this.handler = handler;
    }

    setRequestOptions(options) {
        this.defaultRequestOptions = {
            ...this.defaultRequestOptions,
            ...options,
            headers: {
                ...this.defaultRequestOptions.headers,
                ...options.headers
            }
        };
    }

    hasMessageWithCode(code) {
        return this.queue.some(item => {
            return item.options.headers && item.options.headers.code === code;
        });
    }

    enqueue(message, options = {}) {
        const mergedOptions = {
            ...this.defaultRequestOptions,
            ...options,
            headers: {
                ...this.defaultRequestOptions.headers,
                ...options.headers
            }
        };

        const code = mergedOptions.headers.code;

        if (code && this.hasMessageWithCode(code)) {
            console.log(`消息队列中已存在code为${code}的消息，已忽略重复消息`);
            return Promise.reject(new Error(`重复的消息code: ${code}`));
        }

        return new Promise((resolve, reject) => {
            const queueItem = {
                body: message,
                options: mergedOptions,
                resolve,
                reject
            };

            this.queue.push(queueItem);

            if (!this.processing) {
                this.processNext();
            }
        });
    }

    async processNext() {
        if (this.queue.length === 0 || !this.handler) {
            this.processing = false;
            return;
        }

        this.processing = true;
        const { body, options, resolve, reject } = this.queue.shift();

        // 添加调试日志
        console.log(`开始处理请求: ${options.url || this.defaultRequestOptions.url}, 超时时间: ${options.timeout || this.timeout}ms`);

        try {
            // 使用Promise.race()实现超时控制
            const result = await Promise.race([
                // 正常处理请求
                this.handler(body, options),
                // 超时Promise
                new Promise((_, timeoutReject) => {
                    setTimeout(() => {
                        timeoutReject(new Error(`请求超时（超过${this.timeout/1000}秒）`));
                    }, this.timeout);
                })
            ]);
            resolve(result);
        } catch (error) {
            console.error('Error processing message:', error);
            reject(error);
        } finally {
            this.processNext();
        }
    }

    getLength() {
        return this.queue.length;
    }

    clear() {
        this.queue = [];
    }

    replaceMessage(message, options = {}) {
        const mergedOptions = {
            ...this.defaultRequestOptions,
            ...options,
            headers: {
                ...this.defaultRequestOptions.headers,
                ...options.headers
            }
        };

        const code = mergedOptions.headers.code;
        if (!code) {
            throw new Error('replaceMessage requires a code in headers');
        }

        const index = this.queue.findIndex(item =>
            item.options.headers && item.options.headers.code === code
        );

        if (index !== -1) {
            this.queue.splice(index, 1, {
                body: message,
                options: mergedOptions,
                resolve: (result) => {}, // 占位，实际使用时会被替换
                reject: (error) => {}    // 占位，实际使用时会被替换
            });
            console.log(`已替换code为${code}的消息`);
        } else {
            this.queue.push({
                body: message,
                options: mergedOptions,
                resolve: (result) => {},
                reject: (error) => {}
            });
            console.log(`code为${code}的消息不存在，已添加新消息`);
        }

        if (!this.processing) {
            this.processNext();
        }
    }
}