/**
 * 简化的数据压缩工具
 * 使用 pako 库进行 gzip 压缩
 */

// 如果 pako 库不可用，则提供一个基础的压缩实现
const CompressionUtils = {
    /**
     * 压缩数据
     * @param {string} data - 要压缩的字符串数据
     * @returns {string|Uint8Array} 压缩后的数据
     */
    compress: function(data) {
        // 如果有 pako 库，使用它进行压缩
        if (typeof pako !== 'undefined' && pako.gzip) {
            try {
                // 将字符串转换为 Uint8Array
                const dataUint8Array = new TextEncoder().encode(data);
                // 使用 gzip 压缩
                const compressed = pako.gzip(dataUint8Array);
                console.log(`数据压缩完成: ${data.length} -> ${compressed.length} 字节`);
                return compressed;
            } catch (error) {
                console.error('压缩失败，使用原始数据:', error);
                return data;
            }
        } else {
            // 如果没有 pako 库，返回原始数据
            console.log('未找到 pako 库，使用原始数据');
            return data;
        }
    },

    /**
     * 检查数据是否需要压缩
     * @param {string} data - 要检查的数据
     * @returns {boolean} 是否需要压缩
     */
    shouldCompress: function(data) {
        // 如果数据大于 1KB，则建议压缩
        return data && data.length > 1024;
    },

    /**
     * 获取压缩后的数据和相应的头部信息
     * @param {string} data - 原始数据
     * @returns {object} 包含数据和头部信息的对象
     */
    getCompressedData: function(data) {
        if (this.shouldCompress(data)) {
            const compressed = this.compress(data);
            
            // 如果压缩成功并返回了 Uint8Array
            if (compressed instanceof Uint8Array) {
                // 将 Uint8Array 转换为 base64 字符串
                const base64Data = this.arrayBufferToBase64(compressed);
                return {
                    data: base64Data,
                    headers: {
                        'Content-Encoding': 'gzip',
                        'X-Original-Content-Length': data.length,
                        'X-Compressed-Content-Length': compressed.length
                    }
                };
            } else {
                // 压缩失败，返回原始数据
                return {
                    data: data,
                    headers: {
                        'X-Original-Content-Length': data.length
                    }
                };
            }
        } else {
            // 数据太小，不需要压缩
            return {
                data: data,
                headers: {
                    'X-Original-Content-Length': data.length
                }
            };
        }
    },

    /**
     * 将 ArrayBuffer 转换为 base64 字符串
     * @param {Uint8Array} arrayBuffer - 要转换的 ArrayBuffer
     * @returns {string} base64 字符串
     */
    arrayBufferToBase64: function(arrayBuffer) {
        let binary = '';
        const bytes = new Uint8Array(arrayBuffer);
        const len = bytes.byteLength;
        for (let i = 0; i < len; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return btoa(binary);
    }
};

// 导出模块（如果在模块环境中）
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CompressionUtils;
}